﻿/*
 * (c) ELCUS, 2003, 2005, 2010,2014,2016,2017,2019
 *
 * Part of this code comes from the book "Linux Device Drivers"
 * by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.
*/




#define	UCHAR	 __u8			//unsigned char
#define	USHORT	 __u16
#define	ULONG	 __u32
#define	PULONG	 __u32*			//unsigned long int *
#define	DWORD	 __u32			//long int
#define	BOOL	 bool

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/cdev.h>
#include <linux/unistd.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h> 
#include <linux/ioctl.h> 
#include <linux/version.h>


//#include <linux/sys/types.h> 
#include <linux/sched/signal.h>

#include "pci429LNX.h"


#ifndef CONFIG_PCI
#  error "This driver needs PCI support to be available"
#endif

MODULE_LICENSE("GPL");
MODULE_AUTHOR("VERA KURIANOVA");

#define DEVICE_NUMBER 8

#define VID 0x10b5
#define PID 0x9030
#define SUBVID 0xe1c5
#define SUBPID 0x0201

//================================================================================


typedef struct
{
	 unsigned short int 	p,
				rfi,
				//rfiINT ,
				bSN ,
				PLX ,
				
		       		sigN,
		       		busy,
				numSI,
				numSO,
				codeRID,
				codeRM,
				codeDO,
				codeRFI,
				globChan[8],
				globAddr[8],
				globChanNumber
				;
	 unsigned char		r;

	 int			hSEM;
	 struct cdev  cDev;
	 struct task_struct * pp;
	 dev_t currentDevNum;	
	 unsigned int minorDevNumber;
}DEVICE_Info,*pDEVICE_Info;

static DEVICE_Info aDeviceInfo[DEVICE_NUMBER];
static	struct class* devClass;
static	dev_t	firstDevNum;
spinlock_t deviceLock, deviceIRQlock;
const char * deviceNameStr = "pci429_";
const char * driverNameStr = "pci429LNX: ";
const char * classNameStr = "pci429class";

//================================================================================

static irqreturn_t irq_handler(int irq, 
                 void *dev_id)
{ 
	DEVICE_Info * dev;
	spin_lock(&deviceIRQlock);
	dev = dev_id;

	if (inw(dev->PLX)&4)
	{	
		dev->codeRFI = inw(dev->rfi);
		send_sig((int)dev->sigN,dev->pp,0);
		spin_unlock(&deviceIRQlock);
		return IRQ_HANDLED; 	
	}
	spin_unlock(&deviceIRQlock);
	return IRQ_NONE; 
}

//================================================================================

static int thisModule_open(struct inode *inode, struct file *filp)
{
  	unsigned int i;
  	int result;
	pDEVICE_Info dev;

	spin_lock(&deviceLock);
	i = MINOR(inode->i_rdev);
	aDeviceInfo[i].minorDevNumber = i;

	if(aDeviceInfo[i].busy==0)
	{
		aDeviceInfo[i].busy= 1;
		try_module_get(THIS_MODULE);
     		result = request_irq(aDeviceInfo[i].r,irq_handler,IRQF_SHARED, deviceNameStr, &aDeviceInfo[i]);

		dev = &aDeviceInfo[i];

		filp->private_data = dev;
		spin_unlock(&deviceLock);
		printk ("open: %s%d  serial# %d\n", deviceNameStr, i,aDeviceInfo[i].bSN);

		return 0;  
	}
	else
	{
		spin_unlock(&deviceLock);
		return -1;  
	}
}

//================================================================================

static int thisModule_release(struct inode *inode, struct file *filp)
{

  unsigned int i;
	spin_lock(&deviceLock);

	i = MINOR(inode->i_rdev);

	printk ("release: %s%d  serial# %d\n", deviceNameStr, i,aDeviceInfo[i].bSN);

	free_irq( aDeviceInfo[i].r,&aDeviceInfo[i]);
	aDeviceInfo[i].busy= 0;
	module_put(THIS_MODULE);

	spin_unlock(&deviceLock);
	return 0;
}

//================================================================================

long thisModule_ioctl (struct file *filp,
         unsigned int cmd, unsigned long arg)

{
  unsigned short int port,portA, portD, portRID,
  			tmp, rmod, addr, ii, iii, work,
			chanNumber, arrayNumber,
			arrayDimension, period, workL, workH,
			InterrMask, paramNumber,numberB;
 
  unsigned int devNumber, i;

	pDEVICE_Info dev = (pDEVICE_Info)filp->private_data;

	spin_lock(&deviceLock);

	devNumber = dev->minorDevNumber;
		
 	port = aDeviceInfo[devNumber].p;

		
switch(cmd) 
 {
	case LNX_PCI429_INIT:
	{
		portRID = port+4;
		portD = port+2;

		tmp = inw(portRID);
		aDeviceInfo[devNumber].codeRID=tmp;
		aDeviceInfo[devNumber].numSI=(tmp&0xf00)>>8;
		aDeviceInfo[devNumber].numSO=(tmp&0xf0)>>4;
		
		outw( 0x10, port);
		for (i=0x10;i<0x8000;i++)
		{
			outw( 0, portD);
		}
		
		outw( 0, port);
		for (i=0;i<0x10;i++)
		{
			__get_user(tmp, (USHORT *)(arg+i*2));
			outw( tmp, portD);
		}

		aDeviceInfo[devNumber].codeRM=0xf80;
		outw( 0xf80, portRID);

		__put_user( aDeviceInfo[devNumber].numSI,(USHORT *)(arg));
		__put_user( aDeviceInfo[devNumber].numSO,(USHORT *)(arg+2));

		aDeviceInfo[devNumber].codeDO = 0;

		spin_unlock(&deviceLock);
		break;
		}

	case LNX_PCI429_STOP:
	{
		inw(port+4);

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_INIT_INT:
	{
		__get_user(aDeviceInfo[devNumber].hSEM,(int *)(arg));
		#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
		aDeviceInfo[devNumber].pp = find_task_by_pid(aDeviceInfo[devNumber].hSEM);
		#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,6,31)
		aDeviceInfo[devNumber].pp = find_task_by_vpid(aDeviceInfo[devNumber].hSEM);
		#else
		aDeviceInfo[devNumber].pp = pid_task(find_vpid(aDeviceInfo[devNumber].hSEM), PIDTYPE_PID );
		#endif
		aDeviceInfo[devNumber].sigN=SIGUSR1;

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_RFI_INT:
	{
		__put_user( (aDeviceInfo[devNumber].codeRFI>>12)&1,(USHORT *)(arg));
		__put_user( (aDeviceInfo[devNumber].codeRFI>>8 )&1,(USHORT *)(arg+2));
		__put_user( (aDeviceInfo[devNumber].codeRFI>>9 )&1,(USHORT *)(arg+4));
		__put_user( (aDeviceInfo[devNumber].codeRFI>>10)&1,(USHORT *)(arg+6));
		__put_user( (aDeviceInfo[devNumber].codeRFI>>11)&1,(USHORT *)(arg+8));

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_GET_INT2:
	{
		tmp = inw(port+6);
		__put_user( tmp,(USHORT *)(arg));


		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_RESET_INT:
	{
		outw( 0, port+6);

		spin_unlock(&deviceLock);
		break;
	}


	case LNX_PCI429_SO_O_PARAM:
	{ 
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayNumber, (USHORT *)(arg+2));
			if((arrayNumber > 2)||(arrayNumber<1)) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
				__get_user(arrayDimension, (USHORT *)(arg+6));
				if((arrayDimension>256)||(arrayDimension<1))  arrayDimension =256;
				chanNumber -=1;	
				portD = port + 2;

				outw( 0x5000+0x2000*(arrayNumber-1)+0x200*chanNumber, port);
				for (i=0;i<arrayDimension*2;i++)
				{
					__get_user(tmp, (USHORT *)(arg+8+i*2));
					outw( tmp, portD);
				}

				work = arrayDimension-1;				
				outw( 0x2800+0x1000*(arrayNumber-1)+0x100*chanNumber, port);
				for (i=0;i<work;i++)
				{
					outw( 0x1000+0x200*chanNumber+i*2, portD);
				}

				__get_user(tmp, (USHORT *)(arg+4));
				tmp = tmp<<15;
				outw( 0x1000+0x200*chanNumber+work*2+1+tmp, portD);
				
				__put_user( 0,(USHORT *)(arg));
			}
		}



		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_O_PUSK:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayNumber, (USHORT *)(arg+2));
			if((arrayNumber > 2)||(arrayNumber<1)) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
				chanNumber -=1;	
				arrayNumber -=1;	

				arrayNumber = arrayNumber<<12;

				portD = port + 2;


				outw( 0x28+chanNumber, port);
				outw( 0x0700|(arrayNumber<<1)|arrayNumber, portD);
				
				__put_user( 0,(USHORT *)(arg));
			}
		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_STATE:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)) 
		{
			__put_user(0xffff,(USHORT *)(arg));
		}
		else
		{
			outw(chanNumber+0x17, port);
			work=inw(port+2);
				switch (work& 0x100)
				{
					case 0x0:
						__put_user(0,(USHORT *)(arg));
						break;
					case 0x100:
						__put_user(1,(USHORT *)(arg));
						break;
				}

			__put_user( work&0xff,(USHORT *)(arg+2));
		}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_C_PUSK:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayDimension, (USHORT *)(arg+2));
			if((arrayDimension>256)||(arrayDimension<1))  arrayDimension =256;

			__get_user(period, (USHORT *)(arg+4));
			if((period >256)||(period<arrayDimension)) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
				chanNumber -=1;	
				portD = port + 2;

				__get_user(InterrMask, (USHORT *)(arg+6));
				InterrMask = (InterrMask&1)<<15;

				outw( 0x5000+0x200*chanNumber, port);
				for (i=0;i<arrayDimension*2;i++)
				{
					__get_user(tmp, (USHORT *)(arg+8+i*2));
					outw( tmp, portD);
				}

				for (i=arrayDimension*2;i<512;i++)
				{
					outw( 0, portD);
				}

				outw( 0x2800+0x100*chanNumber, port);
				for (i=0;i<period;i++)
				{
					outw( 0x1000+0x200*chanNumber+i*2, portD);
				}

	
				tmp = 0x2800+0x100*chanNumber + period-1;
				outw( tmp, port);
				work=inw(port+2)+1;
				outw( tmp, port);
				outw( work, port+2);

				tmp = 0x2800+0x100*chanNumber + arrayDimension-1;
				outw( tmp, port);
				work=inw(port+2)+InterrMask;
				outw( tmp, port);
				outw( work, port+2);
	
				outw( (chanNumber+0x28), port);
				outw( 0x0500, port+2);

				__put_user( 0,(USHORT *)(arg));
			}

		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_STOP:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			outw( chanNumber+0x27, port);
			outw( 0x0400, port+2);
			
			__put_user( 0,(USHORT *)(arg));
	}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_C_PARAM:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(paramNumber, (USHORT *)(arg+2));
			if(paramNumber > 255) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
					portD = port + 2;
					addr = 0x5000 + (chanNumber-1)*0x200 + (paramNumber)*2;
					
					outw(addr+1, port);
					outw(0, portD);

					outw(addr, port);
					__get_user(work, (USHORT *)(arg+4));
					outw(work, portD);
					__get_user(work, (USHORT *)(arg+6));
					outw(work, portD);
					__put_user( 0,(USHORT *)(arg));
			}
		}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_G_PARAM_B:
	{
		__get_user(tmp, (USHORT *)(arg));
		if((tmp >(aDeviceInfo[devNumber].globChanNumber*256-1)) || (tmp<1))
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			for(i = 0; i<tmp; i++)
			{
				numberB = i >> 8;

				portD = port + 2;

				addr = 0x5000 + aDeviceInfo[devNumber].globAddr[numberB] + (i&0xff)*2;
				outw(addr, port);

				__get_user(work, (USHORT *)(arg+4+i*4));
				outw(work, portD);
				__get_user(work, (USHORT *)(arg+6+i*4));
				outw(work, portD);
			}
				
			addr = 0x5000+aDeviceInfo[devNumber].globAddr[aDeviceInfo[devNumber].globChanNumber-1]+0x1ff;
			outw(addr, port);
			outw(0, portD);

			__put_user( 0,(USHORT *)(arg));
		}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_G_CHANAL:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			for( i = 0; i<8; i++)
				aDeviceInfo[devNumber].globChan[i] = 0;

			for(i = 1; i<=chanNumber; i++)
			{
				__get_user(tmp, (USHORT *)(arg+i*2));
				if((tmp<1)||(tmp>aDeviceInfo[devNumber].numSO)||(aDeviceInfo[devNumber].globChan[tmp-1] == 1 ))
				{
					__put_user( 1,(USHORT *)(arg));
					i=100;
				}
				else
				{
					aDeviceInfo[devNumber].globChan[tmp-1] = 1;
					aDeviceInfo[devNumber].globAddr[i-1] = (tmp-1)*0x200;
					aDeviceInfo[devNumber].globChanNumber = chanNumber;
					__put_user( 0,(USHORT *)(arg));
				}
			}
		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_G_PARAM:
	{

		__get_user(tmp, (USHORT *)(arg));
		if(tmp >(aDeviceInfo[devNumber].globChanNumber*256-2))
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			numberB = tmp >> 8;				
	
			portD = port + 2;
			
			addr = 0x5000 + aDeviceInfo[devNumber].globAddr[numberB] + (tmp&0xff)*2;
			outw(addr+1, port);
			outw(0, portD);

			outw(addr, port);
			__get_user(work, (USHORT *)(arg+4));
			outw(work, portD);
			__get_user(work, (USHORT *)(arg+6));
			outw(work, portD);


			__put_user( 0,(USHORT *)(arg));
		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SO_G_PUSK:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSO)||(chanNumber<1)||(!aDeviceInfo[devNumber].globChan[chanNumber-1])) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayDimension, (USHORT *)(arg+2));
			if((arrayDimension>256)||(arrayDimension<1)) 
			{
				__put_user( 4,(USHORT *)(arg));
			}
			else
			{
				__get_user(period, (USHORT *)(arg+4));
				if((period >256)||(period<arrayDimension)) 
				{
					__put_user( 2,(USHORT *)(arg));
				}
				else
				{
					chanNumber -=1;	
					portD = port + 2;

					__get_user(InterrMask, (USHORT *)(arg+6));
					InterrMask = (InterrMask&1)<<15;

					outw( 0x2800+0x100*chanNumber, port);
					for (i=0;i<arrayDimension;i++)
					{
						__get_user(paramNumber, (USHORT *)(arg+8+i*2));
						numberB = (paramNumber &0xff00)>> 8;

						if(numberB > (aDeviceInfo[devNumber].globChanNumber-1)) 
						{
							__put_user( 3,(USHORT *)(arg));
							i = 0x1fff;
						}
						else
						{
							addr = 0x1000 +aDeviceInfo[devNumber].globAddr[numberB] + (paramNumber&0xff)*2;
							outw(addr, portD);
						}
					}
					if(i<0x1fff)
					{
						for( i = arrayDimension; i<period; i++)
						{
							tmp = 0x1000+ aDeviceInfo[devNumber].globAddr[aDeviceInfo[devNumber].globChanNumber-1]+0x1fe;
							outw(tmp, portD);
						}

						tmp = 0x2800+0x100*chanNumber + period-1;
						outw( tmp, port);
						work=inw(portD)+1;
						outw( tmp, port);
						outw( work, portD);
	

						tmp = 0x2800+0x100*chanNumber + arrayDimension-1;
						outw( tmp, port);
						work=inw(port+2)+InterrMask;
						outw( tmp, port);
						outw( work, portD);

						outw( (chanNumber+0x28), port);
						outw( 0x0500, portD);
						
						__put_user( 0,(USHORT *)(arg));
					}
				}
			}
		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SI_A_PUSK:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSI)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayNumber, (USHORT *)(arg+2));
			if((arrayNumber > 2)||(arrayNumber<1)) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
				chanNumber -=1;	
				arrayNumber -=1;	

				portD = port + 2;
		
				outw( 0x4000+0x2000*arrayNumber+0x200*chanNumber, port);
				for (i=0;i<512;i++)
				{
					outw( 0, portD);
				}
		
				outw( 0x2000+0x1000*arrayNumber+0x100*chanNumber, port);
				for (i=0;i<256;i++)
				{
					outw( 0x200*chanNumber+i*2, portD);
				}

				__get_user(work, (USHORT *)(arg+4));	//InterrParamAddr
				if(work <= 0xff)
				{
					outw( 0x2000+0x1000*arrayNumber+0x100*arrayNumber+work, port);
					outw( 0x8000+0x200*chanNumber+work*2, portD);
				}

				__get_user(work, (USHORT *)(arg+6));	//StopParamAddr
				if(work <= 0xff)
				{
					addr = 0x2000+0x1000*arrayNumber+0x100*chanNumber+work;
					outw( addr, port);
					tmp=inw(portD)+1;
					outw( addr, port);
					outw( tmp, portD);
				}


				arrayNumber=arrayNumber<<12;
				outw( 0x20+chanNumber, port);
				outw( 0x0500+(arrayNumber<<1)+arrayNumber, portD);
				
				__put_user( 0,(USHORT *)(arg));
			}
		}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SI_STOP:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSI)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			outw( chanNumber+0x1f, port);
			outw( 0x0400, port+2);
			
			__put_user( 0,(USHORT *)(arg));
	}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SI_F_PUSK:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSI)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayNumber, (USHORT *)(arg+2));
			if((arrayNumber > 2)||(arrayNumber<1)) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
				chanNumber -=1;	
				arrayNumber -=1;	

				portD = port + 2;
		
				outw( 0x4000+0x2000*arrayNumber+0x200*chanNumber, port);
				for (i=0;i<512;i++)
				{
					outw( 0, portD);
				}
		
				outw( 0x2000+0x1000*arrayNumber+0x100*chanNumber, port);
				for (i=0;i<256;i++)
				{
					outw( 0x200*chanNumber, portD);
				}

				__get_user(work, (USHORT *)(arg+4));	//InterrParamAddr
				if(work <= 0xff)
				{
					outw( 0x2000+0x1000*arrayNumber+0x100*arrayNumber+work, port);
					outw( 0x8000+0x200*chanNumber, portD);
				}

				__get_user(work, (USHORT *)(arg+6));	//StopParamAddr
				if(work <= 0xff)
				{
					addr = 0x2000+0x1000*arrayNumber+0x100*chanNumber+work;
					outw( addr, port);
					tmp=inw(portD)+1;
					outw( addr, port);
					outw( tmp, portD);
				}


				arrayNumber=arrayNumber<<12;
				outw( 0x20+chanNumber, port);
				outw( 0x0700+(arrayNumber<<1)+arrayNumber, portD);
				
				__put_user( 0,(USHORT *)(arg));
			}
		}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SI_STATE:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSI)||(chanNumber<1)) 
		{
			__put_user(0xffff,(USHORT *)(arg));
		}
		else
		{
			outw(chanNumber+0xf, port);
			work=inw(port+2);
			
			switch (work& 0x100)
			{
				case 0x0:
					__put_user(0,(USHORT *)(arg));
					break;
				case 0x100:
					__put_user(1,(USHORT *)(arg));
					break;
			}

			__put_user( work&0xff,(USHORT *)(arg+2));
		}

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SI_PARAM:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSI)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayNumber, (USHORT *)(arg+2));
			if((arrayNumber > 2)||(arrayNumber<1)) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
				__get_user(paramNumber, (USHORT *)(arg+4));
				if(paramNumber > 255) 
				{
					__put_user( 3,(USHORT *)(arg));
				}
				else
				{
					portD = port + 2;
					addr=0x4000+(arrayNumber-1)*0x2000+(chanNumber-1)*0x200+paramNumber*2;	//
					work = chanNumber + 0xf;
					
					outw(work, port);
					ii=inw(portD)&0xff;

					do
					{
						iii = ii;
						outw(addr, port);
						workL =inw(portD);   
						workH =inw(portD);  
				
						outw(work, port);
						ii=inw(portD)&0xff;

					}while (iii!=ii);
						
					outw(addr+1, port);
					outw(0, portD);
				
					__put_user( 0,(USHORT *)(arg));
					__put_user( workL,(USHORT *)(arg+4));
					__put_user( workH,(USHORT *)(arg+6));
				}
			}
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNX_PCI429_SI_BUFER:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if((chanNumber > aDeviceInfo[devNumber].numSI)||(chanNumber<1)) 
		{
			__put_user( 1,(USHORT *)(arg));
		}
		else
		{
			__get_user(arrayNumber, (USHORT *)(arg+2));
			if((arrayNumber > 2)||(arrayNumber<1)) 
			{
				__put_user( 2,(USHORT *)(arg));
			}
			else
			{
				portD = port + 2;
		
				outw(0x4000+(arrayNumber-1)*0x2000+(chanNumber-1)*0x200, port);
				for (i=0;i<512;i++)
				{
					work=inw(portD);
					__put_user( work,(USHORT *)(arg+4+i*2));
				}
				__put_user( 0,(USHORT *)(arg));
			}
		}

		spin_unlock(&deviceLock);
		break;
	}


	case LNX_PCI429_GET_SN:
	{
		__put_user( aDeviceInfo[devNumber].bSN,(USHORT *)(arg));
	
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_GET_PARAM:
	{
		__put_user( aDeviceInfo[devNumber].PLX-0x4c,(USHORT *)(arg));
		__put_user( port,(USHORT *)(arg+2));
		__put_user( aDeviceInfo[devNumber].r,(USHORT *)(arg+4));

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_GET_DI:
	{
		work=inw(port+8);
		__put_user( work,(USHORT *)(arg));
	
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SET_DO:
	{
		__get_user(work, (USHORT *)(arg));
		if (work<=4)
		{
				work -=1;
				tmp = 1;
				for(i=0; i<work; i++)
					tmp = tmp<<1;

				aDeviceInfo[devNumber].codeDO |= tmp;
				outw(aDeviceInfo[devNumber].codeDO, port+8);
		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_CLR_DO:
	{
		__get_user(work, (USHORT *)(arg));
		if (work<=4)
		{
				work -=1;
				tmp = 0xfffe;
				for(i=0; i<work; i++)
					tmp = (tmp<<1)|1;

				aDeviceInfo[devNumber].codeDO &= tmp;
				outw(aDeviceInfo[devNumber].codeDO, port+8);
		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_SET_DI_INTR:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if (chanNumber<=4)
		{
				chanNumber -=1;
				__get_user(work, (USHORT *)(arg+2));

				if(work==0)
				{
					tmp = 0xeeff;
					for( i=0; i<chanNumber; i++)
						tmp = (tmp<<1)|1;

					aDeviceInfo[devNumber].codeRM &= tmp;
				}
				else
				{
					tmp = 0xfeff;
					for( i=0; i<chanNumber; i++)
						tmp = (tmp<<1)|1;
					aDeviceInfo[devNumber].codeRM &= tmp;

					tmp = 0x1000;
					for( i=0; i<chanNumber; i++)
						tmp = tmp<<1;

					aDeviceInfo[devNumber].codeRM |= tmp;
				}
				outw(aDeviceInfo[devNumber].codeRM, port+4);
		}
		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_STOP_INT:
	{

		spin_unlock(&deviceLock);
		break;
	}

	case LNX_PCI429_CLEAR_DI_INTR:
	{
		__get_user(chanNumber, (USHORT *)(arg));
		if (chanNumber<=4)
		{
				chanNumber -=1;
				tmp = 0x100;
				for(i=0; i<chanNumber; i++)
					tmp = tmp<<1;

				aDeviceInfo[devNumber].codeRM |= tmp;
				outw(aDeviceInfo[devNumber].codeRM, port+4);
		}
		spin_unlock(&deviceLock);
		break;
	}


//============================================================================================
	case IOCTL_INT:
		{

		__get_user(aDeviceInfo[devNumber].hSEM,(int *)(arg+0x1900*2));
		#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
		aDeviceInfo[devNumber].pp = find_task_by_pid(aDeviceInfo[devNumber].hSEM);
		#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,6,31)
		aDeviceInfo[devNumber].pp = find_task_by_vpid(aDeviceInfo[devNumber].hSEM);
		#else
		aDeviceInfo[devNumber].pp = pid_task(find_vpid(aDeviceInfo[devNumber].hSEM), PIDTYPE_PID );
		#endif
		aDeviceInfo[devNumber].sigN=SIGUSR1;

		spin_unlock(&deviceLock);
		break;
		}
		
	case IOCTL_RJ:
		{
		__put_user( port,(unsigned short *)(arg+0x2000));
		__put_user( port+2,(unsigned short *)(arg+0x2002));
		__put_user( port+4,(unsigned short *)(arg+0x2004));
		__put_user( port+6,(unsigned short *)(arg+0x2006));
		__put_user( port+8,(unsigned short *)(arg+0x2008));
		__put_user( port+10,(unsigned short *)(arg+0x200a));

		outw( 0, port);
		port+=2;
		for (i=0;i<0x8000;i++)
		{
			outw( 0, port);
		}

		port+=2;
		outw( 7, port);
		__put_user( 7,(unsigned short *)(arg+0x2200));

		tmp=inw(port);
		__put_user ( tmp,(unsigned short *)(arg+0x2202));
		
		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_INIT: 
		{
		__get_user(port, (unsigned short *)(arg+0x2004));
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));
		__get_user(rmod, (unsigned short *)(arg+0x2200));
		
		rmod=rmod|0x1;

		outw( rmod, port);

		
		outw( 0, portA);
		for (i=0;i<0x1000;i++)
		{
			outw( 0, portD);
		}

		outw( 0, portA);

		for (i=0;i<0x10;i++)
		{
			__get_user(work, (unsigned short *)(arg+i*2));
			outw( work, portD);
		}

		addr=0;
		for (i=0;i<8; i++)
		{		
			__get_user(work, (unsigned short *)(arg+0x2020+i*2));
			addr+=work;
		}
		addr=addr*0x200;
		for  (i=0;i<8; i++)
		{
		__get_user(work, (unsigned short *)(arg+0x2020+i*2));
			if ( work==0)
			{
			__put_user ( addr,(unsigned short *)(arg+0x2030+i*2));
				addr+=0x200;
			}
			else
			{
			__put_user ( 0,(unsigned short *)(arg+0x2030+i*2));
			}
		}

		outw( rmod&0xfffe, port);


		spin_unlock(&deviceLock);
		break;
		}

 	case IOCTL_STOP:
		{
		__get_user(port, (unsigned short *)(arg+0x2004));
		outw( 7, port);
		__put_user( 7,(unsigned short *)(arg+0x2200));

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_DAN_SO:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(addr, (unsigned short *)(arg+0x2220));
		outw( addr, portA);

		__get_user(work, (unsigned short *)(arg+addr*2));
		outw( work, portD);

		addr++;
		__get_user(work, (unsigned short *)(arg+addr*2));
		outw( work, portD);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_WRITE_PRM_G:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(addr, (unsigned short *)(arg+0x2220));
		outw( addr+1, portA);
		outw( 0, portD);

		outw( addr, portA);

		__get_user(work, (unsigned short *)(arg+addr*2));
		outw( work, portD);

		addr++;
		__get_user(work, (unsigned short *)(arg+addr*2));
		outw( work, portD);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_READ_PRM_SS:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(addr, (unsigned short *)(arg+0x2220));
		__get_user(work, (unsigned short *)(arg+0x2218));
		
		work=work+0xf;	//
		outw( work, portA);

		ii=inw(portD)&0xff;
	
		do 
		{
			i=ii;
			outw( addr, portA);
			tmp=inw(portD);
			__put_user ( tmp,(unsigned short *)(arg+addr*2));
			tmp=inw(portD);
			__put_user ( tmp,(unsigned short *)(arg+addr*2+2));

			outw( work, portA);
			ii=inw(portD)&0xff;
		}while (i!=ii);

		spin_unlock(&deviceLock);
		break;
		}



	case IOCTL_READ_PRM_BT:  
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(addr, (unsigned short *)(arg+0x2220));
		outw( addr, portA);

		work=inw(portD);

		do
		{
			__put_user ( work,(unsigned short *)(arg+addr*2));
			outw( addr, portA);
			outw( 0, portD);
			work=inw(portD);
			__put_user ( work,(unsigned short *)(arg+addr*2+2));

			outw( addr, portA);
			work =inw(portD);
		}while (work!=0);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_DAN_SI:  
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(addr, (unsigned short *)(arg+0x2220));
		outw( addr, portA);

		tmp=inw(portD);
		__put_user ( tmp,(unsigned short *)(arg+addr*2));
		tmp=inw(portD);
		__put_user ( tmp,(unsigned short *)(arg+addr*2+2));

		spin_unlock(&deviceLock);
		break;
		}
		
		
	case IOCTL_READ_Z:
		{
		__get_user(port, (unsigned short *)(arg+0x2004));
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(addr, (unsigned short *)(arg+0x2220));
		__get_user(rmod, (unsigned short *)(arg+0x2200));
		i=rmod&0x80;
		outw( addr, portA);
		outw( 0, portD);
		if(i==0)
			work=0x8000;
		else
			work=0;

		outw( work, portD);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_SET_DO:	                    
		{
		__get_user(port, (unsigned short *)(arg+0x2008));
		__get_user(work, (unsigned short *)(arg+0x220c));
		outw( work, port);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_GET_DI:		               
		{
		__get_user(port, (unsigned short *)(arg+0x2008));
		work=inw(port);
		__put_user ( work,(unsigned short *)(arg+0x220e));

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_SET_RI:
		{
		__get_user(port, (unsigned short *)(arg+0x2006));
		outw( 0, port);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_GET_RI:  
		{
		work=inw(aDeviceInfo[devNumber].p+6);

		__put_user ( work,(unsigned short *)(arg+0x2208));

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_GET_RFI:  
		{
		__get_user(port, (unsigned short *)(arg+0x200a));
		work=inw(port);
		__put_user ( work,(unsigned short *)(arg+0x2e00));

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_GET_RFI_INT:  
		{
		__put_user ( aDeviceInfo[devNumber].codeRFI,(unsigned short *)(arg+0x2e02));

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_SET_RD:
		{
		__get_user(port, (unsigned short *)(arg+0x2002));
		__get_user(work, (unsigned short *)(arg+0x220a));
		outw( work, port);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_GET_RD:
		{
		__get_user(port, (unsigned short *)(arg+0x2002));
		work=inw(port);
		__put_user ( work,(unsigned short *)(arg+0x220a));

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_SET_RA:
		{
		__get_user(port, (unsigned short *)(arg+0x2000));
		__get_user(work, (unsigned short *)(arg+0x220a));
		outw( work, port);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_GET_RJ:   
		{
		__get_user(port, (unsigned short *)(arg+0x2004));
		work=inw(port);
		__put_user ( work,(unsigned short *)(arg+0x2202));

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_SET_RM:
		{
		__get_user(port, (unsigned short *)(arg+0x2004));
		__get_user(work, (unsigned short *)(arg+0x2200));
		outw( work, port);

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_READ_MEM:  
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));
		__get_user(addr, (unsigned short *)(arg+0x2210));
		__get_user(work, (unsigned short *)(arg+0x2212));

		outw( addr, portA);

		for (i=0; i<work; i++)
		{
		tmp=inw(portD);
		__put_user ( tmp,(unsigned short *)(arg+addr*2+i*2));
		
		}

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_WRITE_MEM:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));
		__get_user(addr, (unsigned short *)(arg+0x2214));
		__get_user(work, (unsigned short *)(arg+0x2216));


		outw( addr, portA);

		for (i=0; i<work; i++)
		{
		__get_user(tmp, (unsigned short *)(arg+addr*2+i*2));
		outw( tmp, portD);
		}

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_TAB_SO_G:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		
		__get_user(tmp, (unsigned short *)(arg+0x2218));
		
		outw( (tmp+0x17), portA);
		work=(inw(portD))&0x100;		//

		if (work!=0x100)
		{
		addr=0x2800+(tmp-1)*0x100;		//
		__get_user(work, (unsigned short *)(arg+0x2216));
		outw( addr, portA);
		
		for (i=0; i<work; i++)
			{
			__get_user(ii, (unsigned short *)(arg+(addr+i)*2));
			outw( ii, portD);
			}
			
		__put_user ( 0,(unsigned short *)(arg+0x2204));	//not error

		}
		else
		{
		__put_user ( 1,(unsigned short *)(arg+0x2204));	//error
		}

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_TAB_SO_F:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(tmp, (unsigned short *)(arg+0x2218));  //NC
		outw( (tmp+0x17), portA);
		work=(inw(portD));	
			
		i= (work&0x2000)>>13;	// NB  RS
		work=work&0x100;		// ES  RS

		__get_user(ii, (unsigned short *)(arg+0x221a));
		if ((work==0x100)&&(i==ii))	//
		{
			__put_user ( 1,(unsigned short *)(arg+0x2204));	//error
		}
		else
		{
			i=tmp-1;
			addr=0x2800+ii*0x1000+i*0x100;
			__get_user(work, (unsigned short *)(arg+0x2030+i*2));
				
			outw( addr, portA);

			for (i=0; i<256; i++)
			{
			__get_user(tmp, (unsigned short *)(arg+(addr+i)*2)); 
			tmp=tmp|0x1000|i*0x2|work ;
			outw( tmp, portD);
			}
			
			__put_user ( 0,(unsigned short *)(arg+0x2204));	//not error

		}


		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_TAB_SI:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));

		__get_user(tmp, (unsigned short *)(arg+0x2218));  //NC
		outw( (tmp+0xf), portA);
		work=(inw(portD));	

		i= (work&0x2000)>>13;	//  NB 
		work=work&0x100;		//  ES  RS

		__get_user(ii, (unsigned short *)(arg+0x221a));    //nb
		if ((work==0x100)&&(i==ii))	
		{
			__put_user ( 1,(unsigned short *)(arg+0x2204));	//error
		}
		else
		{
			addr=0x2000+ii*0x1000+(tmp-1)*0x100;
			
			outw( addr, portA);

			for (i=0; i<256; i++)
			{
			__get_user(work, (unsigned short *)(arg+(addr+i)*2)); 
			work=work|i*2|((tmp-1)*0x200);
			outw( work, portD);
			
}
			
			__put_user ( 0,(unsigned short *)(arg+0x2204));	//not error

		}


		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_STOP_CH:	 
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));
		
		
		__get_user(work, (unsigned short *)(arg+0x221c));
		work =work+0x20;
		outw( (work), portA);
		outw( 0x400, portD);

		__put_user ( 0x400,(unsigned short *)(arg+work*2));	//

		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_PUSK_SO:	  
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));
		__get_user(addr, (unsigned short *)(arg+0x221c));  //nc
		addr--;
		__get_user(work, (unsigned short *)(arg+0x221e));  //nb


		outw( (addr+0x28), portA);

		i =(inw(portD))&0x400;	//

		if ( i==0x400 )
			__put_user ( 1,(unsigned short *)(arg+0x2204));	//error
		else
		{
			__get_user(i, (unsigned short *)(arg+0x2040+(addr*3+work*24)*2));
			
			if (i==0)
			__put_user ( 2,(unsigned short *)(arg+0x2204));	//error
			else
			{
			outw( (addr+0x28), portA);
			__get_user(tmp, (unsigned short *)(arg+(addr+0x28)*2));  
			outw( tmp, portD);
			__put_user ( 0,(unsigned short *)(arg+0x2204));	//not error

			}
		}


		spin_unlock(&deviceLock);
		break;
		}

	case IOCTL_PUSK_SI:
		{
		__get_user(portA, (unsigned short *)(arg+0x2000));
		__get_user(portD, (unsigned short *)(arg+0x2002));
		__get_user(addr, (unsigned short *)(arg+0x221c));  //nc
		addr--;
		__get_user(work, (unsigned short *)(arg+0x221e));  //nb

		outw( (addr+0x20), portA);
		i =(inw(portD))&0x400;	//

		if ( i==0x400 )
			__put_user ( 1,(unsigned short *)(arg+0x2204));	//error
		else
		{
			__get_user(i, (unsigned short *)(arg+0x20a0+(addr+work
			
			*8)*2));
			
			if (i==0)
			__put_user ( 2,(unsigned short *)(arg+0x2204));	//error
			else
			{
			outw( (addr+0x20), portA);
			__get_user(tmp, (unsigned short *)(arg+(addr+0x20)*2));  
			outw( tmp, portD);
			__put_user ( 0,(unsigned short *)(arg+0x2204));	//not error

			}
		}



		spin_unlock(&deviceLock);
		break;
		}
	case IOCTL_GET_SER_NUMBER:
	
		{
		__put_user ( aDeviceInfo[devNumber].bSN,(unsigned short *)(arg+0x30a0));

		spin_unlock(&deviceLock);
		break;
		}

 }
	return 0;  
}

//================================================================================

static struct file_operations thisModule_fops = {
	.owner	= THIS_MODULE,	
 	.unlocked_ioctl   = thisModule_ioctl,
 	.open  =  thisModule_open,
 	.release = thisModule_release
};

unsigned short ptr, SN[6];

	
static int __init  thisModule_initialization(void)
{
   unsigned long int  ch1,sss;
   unsigned short  m, number = 0, temp;
   int i,err;
	
	int result;
    
	struct pci_dev *dev = NULL;
		
	dev_t currentDevNum;	
	struct device *newDev = NULL;
 
   	spin_lock_init(&deviceLock);
   	spin_lock_init(&deviceIRQlock);

          
          
    	dev = pci_get_device(VID,PID, dev);
 
   	while ((dev)&&(number<DEVICE_NUMBER))
	{
		err = pci_enable_device(dev);

		pci_read_config_word(dev, 0x2c, &ptr);

	    	if (ptr==SUBVID) 
	    	{
			pci_read_config_word(dev, 0x2e, &ptr);
 
	    		if (ptr==SUBPID) 
	    		{

				pci_read_config_word(dev, 0x14, &temp);
				temp = (temp & 0xfffe) + 0x4c;
				aDeviceInfo[number].PLX =  temp;

				pci_read_config_word(dev, 0x18, &temp);	
				temp &=0xfffe;
				aDeviceInfo[number].p =  temp;
 				aDeviceInfo[number].rfi = temp+0xa;

				aDeviceInfo[number].r=dev-> irq;

				ptr=0;
				pci_write_config_word(dev, 0x4e, 0xcc);
	
				m=0; ch1=0;
	
				while(!(m&0x8000))
				{
					pci_read_config_word(dev, 0x4e, (u16*)&m);
					ch1++;
					if(ch1==0xfff)
					{
						ch1=0;
						pci_write_config_word(dev, 0x4e, 0xcc);
					}
				};
	
		
				pci_read_config_dword(dev, 0x50,(u32 *) &sss);

				SN[0]=(sss>>16)&0xff;
				SN[1]=(sss>>24)&0xff;


				pci_write_config_word(dev, 0x4e, 0xd0);
				m=0; ch1=0;
	
				while(!(m&0x8000))
				{
					pci_read_config_word(dev, 0x4e, (u16*)&m);
					ch1++;
					if(ch1==0xfff)
					{
						ch1=0;
						pci_write_config_word(dev, 0x4e, 0xd0);
					}
				};
	
				pci_read_config_dword(dev, 0x50, (u32 *)&sss);

     				SN[2]=sss&0xff;
     				SN[3]=(sss>>8)&0xff;
     				SN[4]=(sss>>16)&0xff;
				aDeviceInfo[number].bSN =(SN[0]-0x30)*10000+(SN[1]-0x30)*1000+(SN[2]-0x30)*100+(SN[3]-0x30)*10+(SN[4]-0x30);
 
				number++;

			}
		}
 	    	dev = pci_get_device(VID,PID, dev);
 	}

	if(!number)
	{
		printk ("%s Sorry, devices were not found\n  initialization failed\n", driverNameStr);
		return -1;
	}


	result = alloc_chrdev_region(&firstDevNum,0,number,driverNameStr);
	if(result<0)
	{
		printk ("%s alloc_chrdev_region failed\n", driverNameStr);
		return -1;
	}
		
	devClass = class_create(THIS_MODULE,classNameStr);
	
	if(devClass==NULL)
	{
		printk ("%s class_create failed\n", driverNameStr);
		unregister_chrdev_region(firstDevNum,1);
		return -1;
	}
	
	
	for(i=0; i<number; i++)
	{
	
		currentDevNum = MKDEV(MAJOR(firstDevNum),MINOR(firstDevNum)+i);
		aDeviceInfo[i].currentDevNum = currentDevNum;
			
		newDev = device_create(devClass,NULL,currentDevNum,NULL,"%s%d",deviceNameStr,i);

		if(newDev==NULL)
		{
			printk ("%s device_create failed\n", driverNameStr);
			class_destroy(devClass);	
			unregister_chrdev_region(currentDevNum,1);
			return -1;
		}

		cdev_init(&aDeviceInfo[i].cDev, &thisModule_fops);
		
		if(cdev_add(&aDeviceInfo[i].cDev,currentDevNum,1)==-1)
		{
			printk ("%s cdev_add failed\n", driverNameStr);
			device_destroy(devClass,currentDevNum);	
			class_destroy(devClass);	
			unregister_chrdev_region(currentDevNum,1);
			return -1;
		}
		if(!i)
			printk ("%s initialization\n", driverNameStr);
		printk ("           /dev/%s%d  serial# %d\n",deviceNameStr,i,aDeviceInfo[i].bSN);
			
	}

	for (i=0; i<number; i++)
		 aDeviceInfo[i].busy= 0;

	for (i=number; i<DEVICE_NUMBER; i++)
		 aDeviceInfo[i].busy= 2;
  
 	return 0; 
}

//================================================================================

static void __exit thisModule_cleanup(void) 
 {  
	 int i;
	 for(i=0; i<DEVICE_NUMBER; i++)
	 {
		if(aDeviceInfo[i].busy != 2)
		{
			cdev_del(&aDeviceInfo[i].cDev);
			device_destroy(devClass,aDeviceInfo[i].currentDevNum);	
		}
	 }
	class_destroy(devClass);	
	unregister_chrdev_region(firstDevNum,1);

	printk ("%s cleanup\n",driverNameStr);
   }
 
//================================================================================

module_init(thisModule_initialization);
module_exit(thisModule_cleanup);
