
/*
 * (c) ELCUS, 2005, 2010,2014,2019
 *
 * Part of this code comes from the book "Linux Device Drivers"
 * by Alessandro Rubini and Jonathan Corbet, published
 * by O'Reilly & Associates.
*/


#define	UCHAR	 __u8			//unsigned char
#define	USHORT	 __u16
#define	ULONG	 __u32
#define	PULONG	 __u32*			//unsigned long int *
#define	DWORD	 __u32			//long int
#define	BOOL	 bool

#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <asm/uaccess.h>
#include <linux/types.h>
#include <linux/pci.h>
#include <linux/cdev.h>
#include <linux/unistd.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h> 
#include <linux/ioctl.h> 
#include <linux/version.h>
#include "tms.h"

#include "pci429-4.h"



#ifndef CONFIG_PCI
#  error "This driver needs PCI support to be available"
#endif


MODULE_LICENSE("GPL");
MODULE_AUTHOR("VERA KURIANOVA");

#define DEVICE_NUMBER 8

#define VID 0x10b5
#define PID 0x9030
#define SUBVID 0xe1c5
#define SUBPID 0x0202

typedef struct
{
 unsigned short int 	RA,
			      	RD,
			      	RID ,
			      	RM ,
			      	RV ,
			      	RDV ,
			      	RDP ,
			      	RD2 ,
			      	RCA ,
			      	RDV1 ,
			      	RDV2 ,
			      	bSN ,
			      	PLX ,
				hSEM,
	       			sigN,
	       			busy;
 unsigned char		      	r;
 	 struct cdev  cDev;
	 struct task_struct * pp;
	 dev_t currentDevNum;	
	 unsigned int minorDevNumber;
}DEVICE_Info,*pDEVICE_Info;

static DEVICE_Info aDeviceInfo[DEVICE_NUMBER];
static	struct class* devClass;
static	dev_t	firstDevNum;
spinlock_t deviceLock, deviceIRQlock;
const char * deviceNameStr = "pci429_4";
const char * driverNameStr = "pci429-4LNX: ";
const char * classNameStr = "pci429-4class";

//================================================================================

static irqreturn_t irq_handler(int irq, 
                 void *dev_id)
{ 
	DEVICE_Info * dev;
	spin_lock(&deviceIRQlock);
	dev = dev_id;

	if ((inw(dev->PLX)&5)==5)
	{	
		outw(0xd53,dev->PLX);
		send_sig(dev->sigN,dev->pp,0);
		spin_unlock(&deviceIRQlock);
		return IRQ_HANDLED; 	
	}
	spin_unlock(&deviceIRQlock);
	return IRQ_NONE; 
}

					 

//================================================================================

static int thisModule_open(struct inode *inode, struct file *filp)
{
  	unsigned int i;
  	int result;
	pDEVICE_Info dev;

	spin_lock(&deviceLock);
	i = MINOR(inode->i_rdev);
	aDeviceInfo[i].minorDevNumber = i;

	if(aDeviceInfo[i].busy==0)
	{
		aDeviceInfo[i].busy= 1;
		try_module_get(THIS_MODULE);
     		result = request_irq(aDeviceInfo[i].r,irq_handler,IRQF_SHARED, deviceNameStr, &aDeviceInfo[i]);

		dev = &aDeviceInfo[i];

		filp->private_data = dev;
		spin_unlock(&deviceLock);
		printk ("open: %s%d  serial# %d\n", deviceNameStr, i,aDeviceInfo[i].bSN);

		return 0;  
	}
	else
	{
		printk ("open: serial# ");
		spin_unlock(&deviceLock);
		return -1;  
	}
}

//================================================================================

static int thisModule_release(struct inode *inode, struct file *filp)
{

  unsigned int i;
	spin_lock(&deviceLock);

	i = MINOR(inode->i_rdev);

	printk ("release: %s%d  serial# %d\n", deviceNameStr, i,aDeviceInfo[i].bSN);

	free_irq( aDeviceInfo[i].r,&aDeviceInfo[i]);
	aDeviceInfo[i].busy= 0;
	module_put(THIS_MODULE);

	spin_unlock(&deviceLock);
	return 0;
}

//================================================================================

long thisModule_ioctl (struct file *filp,
         unsigned int cmd, unsigned long arg)

{
 unsigned short int adr, adrEnd, ind, ch, i1, work=0;
 unsigned int devNumber;

	pDEVICE_Info dev = (pDEVICE_Info)filp->private_data;

spin_lock(&deviceLock);

devNumber = dev->minorDevNumber;

 switch(cmd) 
 {
 	case LNXioctl_PCI429_4_readFL_I:
	{

		__get_user(adr, (unsigned short *)(arg));
		adr = RRP_i_ext + ((adr-1)&0xf)*0x80+6;  
		outw( adr,  aDeviceInfo[devNumber].RA); 

		work=inw( aDeviceInfo[devNumber].RD);
		__put_user ( work,(unsigned short *)(arg+2));

		if (work!=0)
		{
			outw( adr,  aDeviceInfo[devNumber].RA); 
			outw( 0,  aDeviceInfo[devNumber].RD); 
		}
		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readFL_O:
	{

		__get_user(adr, (unsigned short *)(arg));
		adr = RRV_i_ext + ((adr-1)&0xf)*0x80+6;  
		outw( adr,  aDeviceInfo[devNumber].RA); 

		work=inw( aDeviceInfo[devNumber].RD);
		__put_user ( work,(unsigned short *)(arg+2));

		if (work!=0)
		{
			outw( adr,  aDeviceInfo[devNumber].RA); 
			outw( 0,  aDeviceInfo[devNumber].RD); 
		}
		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readPK:
	{
		__get_user(adr, (unsigned short *)(arg));
		outw( adr,  aDeviceInfo[devNumber].RA);

		i1 = inw( aDeviceInfo[devNumber].RD2);
		work=inw( aDeviceInfo[devNumber].RD2);
		__put_user ( i1,(unsigned short *)(arg+2));
		__put_user ( work,(unsigned short *)(arg+4));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writePK:
	{

		__get_user(adr, (unsigned short *)(arg));
		outw( adr,  aDeviceInfo[devNumber].RA);

		__get_user(i1, (unsigned short *)(arg+2));
		__get_user(work, (unsigned short *)(arg+4));
		outw( i1,  aDeviceInfo[devNumber].RD2);
		outw( work,  aDeviceInfo[devNumber].RD2);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_read_BKOP:
	{
	
		outw( UKOP_i_ext,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<5; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_read_KPM:
	{
			outw( CPM0_i_ext,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<4; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_read_KVM:
	{
		outw( CBM0_i_ext,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<4; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_read_KMP:
	{
		outw( CBP0_i_ext,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<4; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_read_KOMP:
	{
		outw( CBOM_i_ext,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<4; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readBU_I:
	{
		__get_user(adr, (unsigned short *)(arg));
		adr = RRP_i_ext + ((adr-1)&0xf)*0x80;  
		outw( adr,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<8; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readBU_O:
	{
		__get_user(adr, (unsigned short *)(arg));
		adr = RRV_i_ext + ((adr-1)&0xf)*0x80;  
		outw( adr,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<8; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_read_KVM_blDat1:
	{
		outw( BPK1_i_ext,  aDeviceInfo[devNumber].RA);

		for(ind=0; ind<512; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_read_KVM_blDat2:
	{
		outw( BPK2_i_ext,  aDeviceInfo[devNumber].RA);

		for(ind=0; ind<512; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeKOP_0:
	{
		outw( UKOP_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RD);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeKOP_1:
	{
		outw( UKOP_i_ext+1,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg+2));
		outw( work,  aDeviceInfo[devNumber].RD);

		outw( UKOP_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RD);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeKOP_2:
	{
		outw( UKOP_i_ext+1,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg+2));
		outw( work,  aDeviceInfo[devNumber].RD);

		__get_user(work, (unsigned short *)(arg+4));
		outw( work,  aDeviceInfo[devNumber].RD);

		outw( UKOP_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RD);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeKOP_3:
	{
		outw( UKOP_i_ext+1,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg+2));
		outw( work,  aDeviceInfo[devNumber].RD);

		__get_user(work, (unsigned short *)(arg+4));
		outw( work,  aDeviceInfo[devNumber].RD);

		__get_user(work, (unsigned short *)(arg+6));
		outw( work,  aDeviceInfo[devNumber].RD);

		outw( UKOP_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RD);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeBU_O:
	{
		__get_user(adr, (unsigned short *)(arg));
		adr = RRV_i_ext + ((adr-1)&0xf)*0x80;  
		outw( adr,  aDeviceInfo[devNumber].RA); 

		__get_user(work, (unsigned short *)(arg+2));
		outw( work,  aDeviceInfo[devNumber].RD);			

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeBU_I:
	{
		__get_user(adr, (unsigned short *)(arg));
		adr = RRP_i_ext + ((adr-1)&0xf)*0x80+2;  
		outw( adr,  aDeviceInfo[devNumber].RA); 

		__get_user(work, (unsigned short *)(arg+4));
		outw( work,  aDeviceInfo[devNumber].RD);		

		adr = adr-2;	
		outw( adr,  aDeviceInfo[devNumber].RA); 

		__get_user(work, (unsigned short *)(arg+2));
		outw( work,  aDeviceInfo[devNumber].RD);		

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_write_KPM_blAdr1:
	{
		outw( BAPMP1_i_ext,  aDeviceInfo[devNumber].RA);

		ind=0;
		do	
		{
		__get_user(work, (unsigned short *)(arg+2*ind));
		outw( work,  aDeviceInfo[devNumber].RD);		
		ind++;
		}while((ind<256) && ((work&0x8000)!=0));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_write_KPM_blAdr2:
	{
		outw( BAPMP2_i_ext,  aDeviceInfo[devNumber].RA);

		ind=0;
		do	
		{
		__get_user(work, (unsigned short *)(arg+2*ind));
		outw( work,  aDeviceInfo[devNumber].RD);		
		ind++;
		}while((ind<256) && ((work&0x8000)!=0));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_write_KPM_blDat1:
	{
		outw( BPMP1_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(ch, (unsigned short *)(arg));
		ch = ch*2;

		for (ind = 1; ind<=ch; ind++)	
		{
			__get_user(work, (unsigned short *)(arg+2*ind));
			outw( work,  aDeviceInfo[devNumber].RD);		
		}


		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_write_KPM_blDat2:
	{
		outw( BPMP2_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(ch, (unsigned short *)(arg));
		ch = ch*2;

		for (ind = 1; ind<=ch; ind++)	
		{
			__get_user(work, (unsigned short *)(arg+2*ind));
			outw( work,  aDeviceInfo[devNumber].RD);		
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_write_KVM_blAdr1:
	{
		outw( BAP1_i_ext,  aDeviceInfo[devNumber].RA);

		ind=0;
		do	
		{
		__get_user(work, (unsigned short *)(arg+2*ind));
		outw( work,  aDeviceInfo[devNumber].RD);		
		ind++;
		}while((ind<256) && ((work&0xC000)!=0));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_write_KVM_blAdr2:
	{
		outw( BAP2_i_ext,  aDeviceInfo[devNumber].RA);

		ind=0;
		do	
		{
		__get_user(work, (unsigned short *)(arg+2*ind));
		outw( work,  aDeviceInfo[devNumber].RD);		
		ind++;
		}while((ind<256) && ((work&0xC000)!=0));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_write_KMP_blAdr:
	{
		outw( BAPP_i_ext,  aDeviceInfo[devNumber].RA);

		ind=0;
		do	
		{
		__get_user(work, (unsigned short *)(arg+2*ind));
		outw( work,  aDeviceInfo[devNumber].RD);		
		ind++;
		}while((ind<256) && (work!=0));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_getSN:
	{
		__put_user (  aDeviceInfo[devNumber].bSN,(unsigned short *)(arg));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_getState:
	{
		outw( CIA_i_ext,  aDeviceInfo[devNumber].RA);

		for	 ( ind=0;ind<4; ind++)
		{
			work=inw( aDeviceInfo[devNumber].RD);
			__put_user ( work,(unsigned short *)(arg+ind*2));
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_IntInit:
	{
		__get_user( aDeviceInfo[devNumber].hSEM,(int *)(arg));
		#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,24)
		aDeviceInfo[devNumber].pp = find_task_by_pid(aDeviceInfo[devNumber].hSEM);
		#elif LINUX_VERSION_CODE < KERNEL_VERSION(2,6,31)
		aDeviceInfo[devNumber].pp = find_task_by_vpid(aDeviceInfo[devNumber].hSEM);
		#else
		aDeviceInfo[devNumber].pp = pid_task(find_vpid(aDeviceInfo[devNumber].hSEM), PIDTYPE_PID );
		#endif
		 aDeviceInfo[devNumber].sigN=SIGUSR1;

		outw( 0xd53,  aDeviceInfo[devNumber].PLX);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_IntRst:
	{
		 aDeviceInfo[devNumber].hSEM = 0;
		 aDeviceInfo[devNumber].sigN = 0;

		outw( 0xd12,  aDeviceInfo[devNumber].PLX);
		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_getPRM:
	{
		__put_user ( ( aDeviceInfo[devNumber].PLX-0x4c),(unsigned short *)(arg));
		__put_user (  aDeviceInfo[devNumber].RA,(unsigned short *)(arg+2));
		__put_user (  aDeviceInfo[devNumber].r,(unsigned short *)(arg+4));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeBKV:
	{
		__get_user(work, (unsigned short *)(arg));

		work = ((work-1)&0xf)*0x400 + bufO_i_ext;
		outw( work,  aDeviceInfo[devNumber].RA);

		__get_user(ch, (unsigned short *)(arg+2));
		ch = ch*2;

		if( ch>0x400) ch = 0x400;

		for(ind = 0;ind<ch;ind++)
			{
			__get_user(work, (unsigned short *)(arg+4+ind*2));
			outw( work, aDeviceInfo[devNumber].RD);
			
			}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeBKV2:
	{
		__get_user(work, (unsigned short *)(arg));

		work = ((work-1)&0xf)*0x400 + bufO_i_ext+0x200;
		outw( work,  aDeviceInfo[devNumber].RA);

		__get_user(ch, (unsigned short *)(arg+2));
		ch = ch*2;

		if( ch>0x200) ch = 0x200;

		for(ind = 0;ind<ch;ind++)
			{
			__get_user(work, (unsigned short *)(arg+4+ind*2));
			outw( work, aDeviceInfo[devNumber].RD);
			
			}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeBG:
	{
		__get_user(adr, (unsigned short *)(arg));
		__get_user(ch, (unsigned short *)(arg+2));
		ch = ch*2;

		if (adr<0xC000)
		{
			__put_user ( 1,(unsigned short *)(arg));
			spin_unlock(&deviceLock);
			break;
		}
	
		if  (((unsigned short)(adr+ch))<adr) ch =(unsigned short)( 0x10000 - adr);
		ind = 0;

		while(ind<ch)
		{
			outw( adr, aDeviceInfo[devNumber].RA);
			adrEnd = (adr&0xFC00)+0x400;

			i1 = 0;

			if ( adr>=0xfc00 )
			{
				while(ind<ch)
				{
					__get_user(work, (unsigned short *)(arg+4+ind*2));
					outw( work, aDeviceInfo[devNumber].RD);
					ind++;
				}
			}
			else
			{
				while((ind<ch) && ((adr+i1)<adrEnd))
				{
					__get_user(work, (unsigned short *)(arg+4+ind*2));
					outw( work, aDeviceInfo[devNumber].RD);
					ind++;
					i1++;
				}
			}
			adr = adrEnd;
		}
		__put_user ( 0,(unsigned short *)(arg));


		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeTab_I:
	{
		__get_user(work, (unsigned short *)(arg));
		work = ((work-1)&0xf)*0x400 + bufI_i_ext;
		outw( work,  aDeviceInfo[devNumber].RA);

		__get_user(ch, (unsigned short *)(arg+2));

		ind = 0;
		while((ind<ch) && (ind<0x400))
		{
			__get_user(work, (unsigned short *)(arg+4+ind*2));
			outw( work,  aDeviceInfo[devNumber].RD);
			ind++;
		}

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeTab_O:
	{
		__get_user(work, (unsigned short *)(arg));
		work = ((work-1)&0xf)*0x400 + bufO_i_ext;
		outw( work,  aDeviceInfo[devNumber].RA);

		__get_user(ch, (unsigned short *)(arg+2));

		ind = 0;
		while((ind<ch) && (ind<0x400))
		{
			__get_user(work, (unsigned short *)(arg+4+ind*2));
			outw( work,  aDeviceInfo[devNumber].RD);
			ind++;
		}

		spin_unlock(&deviceLock);
		break;
	}
	case  LNXioctl_PCI429_4_readRV:
	{
		work=inw( aDeviceInfo[devNumber].RV);
		__put_user ( work,(unsigned short *)(arg));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeRV:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RV);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeRCA:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RCA);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeDV1:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RDV1);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeDV2:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RDV2);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readRA:
	{
		work=inw( aDeviceInfo[devNumber].RA);
		__put_user ( work,(unsigned short *)(arg));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeRA:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RA);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readRD:
	{
		work=inw( aDeviceInfo[devNumber].RD);
		__put_user ( work,(unsigned short *)(arg));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeRD:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RD);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeRM:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RM);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readRDP:
	{
		work=inw( aDeviceInfo[devNumber].RDP);
		__put_user ( work,(unsigned short *)(arg));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeRDV:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RDV);

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readRID:
	{
		work=inw( aDeviceInfo[devNumber].RID);  
		__put_user ( work,(unsigned short *)(arg));

		spin_unlock(&deviceLock);

		break;
	}
	case LNXioctl_PCI429_4_readMem:
	{
		__get_user(adr, (unsigned short *)(arg));
		__get_user(ch, (unsigned short *)(arg+2));

		if  (((unsigned short)(adr+ch))<adr) ch =(unsigned short)( 0x10000 - adr);
		ind = 0;
		
		while(ind<ch)
		{
			outw( adr, aDeviceInfo[devNumber].RA);
			adrEnd = (adr&0xFC00)+0x400;

			i1 = 0;

			if ( adr>=0xfc00 )
			{
				while(ind<ch)
				{
					work=inw( aDeviceInfo[devNumber].RD);
					__put_user ( work,(unsigned short *)(arg+4+ind*2));
					ind++;
				}
			}
			else
			{
				while((ind<ch) && ((adr+i1)<adrEnd))
				{
					work=inw( aDeviceInfo[devNumber].RD);
					__put_user ( work,(unsigned short *)(arg+4+ind*2));
					ind++;
					i1++;
				}
			}
			adr = adrEnd;
		}
		__put_user ( 0,(unsigned short *)(arg));

		spin_unlock(&deviceLock);
			break;
	}
	case LNXioctl_PCI429_4_writeMem:
	{
		__get_user(adr, (unsigned short *)(arg));
		__get_user(ch, (unsigned short *)(arg+2));

		if  (((unsigned short)(adr+ch))<adr) ch =(unsigned short)( 0x10000 - adr);

		ind = 0;

		while(ind<ch)
		{
			outw( adr, aDeviceInfo[devNumber].RA);
			adrEnd = (adr&0xFC00)+0x400;

			i1 = 0;

			if ( adr>=0xfc00 )
			{
				while(ind<ch)
				{
					__get_user(work, (unsigned short *)(arg+4+ind*2));
					outw( work, aDeviceInfo[devNumber].RD);
					ind++;
				}
			}
			else
			{
				while((ind<ch) && ((adr+i1)<adrEnd))
				{
					__get_user(work, (unsigned short *)(arg+4+ind*2));
					outw( work, aDeviceInfo[devNumber].RD);
					ind++;
					i1++;
				}
			}
			adr = adrEnd;
		}
		__put_user ( 0,(unsigned short *)(arg));

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_readWord:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RA);

		work=inw( aDeviceInfo[devNumber].RD);
		__put_user ( work,(unsigned short *)(arg+2));
	

		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_writeWord:
	{
		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg+2));
		outw( work,  aDeviceInfo[devNumber].RD);


		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_setFreq:
	{
		adr = RRP_i_ext + 7;
		for(ind = 0; ind<16; ind++)
		{
			outw( adr,  aDeviceInfo[devNumber].RA);
			__get_user(work, (unsigned short *)(arg+ind*2));
			outw( work,  aDeviceInfo[devNumber].RD);
			adr = adr + 0x80;
		}

	
		adr = RRV_i_ext + 7;
		for(ind = 0; ind<16; ind++)
		{
			outw( adr,  aDeviceInfo[devNumber].RA);
			__get_user(work, (unsigned short *)(arg+ind*2+32));
			outw( work,  aDeviceInfo[devNumber].RD);
			adr = adr + 0x80;
		}


		spin_unlock(&deviceLock);
		break;
	}
	case LNXioctl_PCI429_4_setKCR:
	{
		outw( UCMT_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RD);
			

		spin_unlock(&deviceLock);
		break;
	}		
	case LNXioctl_PCI429_4_setKCI:
	{
		outw( UNI_i_ext,  aDeviceInfo[devNumber].RA);

		__get_user(work, (unsigned short *)(arg));
		outw( work,  aDeviceInfo[devNumber].RD);
			

		spin_unlock(&deviceLock);
		break;
	}		
 }
 return 0;  
}



//================================================================================

static struct file_operations thisModule_fops = {
	.owner	= THIS_MODULE,	
 	.unlocked_ioctl   = thisModule_ioctl,
 	.open  =  thisModule_open,
 	.release = thisModule_release
};

unsigned short ptr, SN[6];

	
static int __init  thisModule_initialization(void)
{
   unsigned long int  ch1,sss;
   unsigned short  m, number = 0, temp;
   int i,err;
	
	int result;
    
	struct pci_dev *dev = NULL;
		
	dev_t currentDevNum;	
	struct device *newDev = NULL;
 
   	spin_lock_init(&deviceLock);
   	spin_lock_init(&deviceIRQlock);


      
    	dev = pci_get_device(VID,PID, dev);
 
    	while ((dev)&&(number<DEVICE_NUMBER))
	{
		err = pci_enable_device(dev);

		pci_read_config_word(dev, 0x2c, &ptr);

	    	if (ptr==SUBVID) 
	    	{
			pci_read_config_word(dev, 0x2e, &ptr);
 
	    		if (ptr==SUBPID) 
	    		{
				pci_read_config_word(dev, 0x14, &temp);
				temp = (temp & 0xfffe) + 0x4c;
				aDeviceInfo[number].PLX =  temp;

				pci_read_config_word(dev, 0x18, &temp);	
				temp &=0xfffe;
				aDeviceInfo[number].RA  =  temp;
				aDeviceInfo[number].RD  =  temp+0x2;
				aDeviceInfo[number].RID =  temp+0x4;
				aDeviceInfo[number].RM  =  temp+0x4;
				aDeviceInfo[number].RV  =  temp+0x6;
				aDeviceInfo[number].RDV =  temp+0x8;
				aDeviceInfo[number].RDP =  temp+0x8;
				aDeviceInfo[number].RD2 =  temp+0xA;
				aDeviceInfo[number].RCA =  temp+0xC;
				aDeviceInfo[number].RDV1 = temp+0xC;
				aDeviceInfo[number].RDV2 = temp+0xE;

				aDeviceInfo[number].r = dev-> irq;
 
				ptr=0;
				pci_write_config_word(dev, 0x4e, 0xcc);
	
				m=0; ch1=0;
	
				while(!(m&0x8000))
				{
					pci_read_config_word(dev, 0x4e, (u16*)&m);
					ch1++;
					if(ch1==0xfff)
					{
						ch1=0;
						pci_write_config_word(dev, 0x4e, 0xcc);
					}
				};
	
		
				pci_read_config_dword(dev, 0x50,(u32 *) &sss);

				SN[0]=(sss>>16)&0xff;
				SN[1]=(sss>>24)&0xff;


				pci_write_config_word(dev, 0x4e, 0xd0);
				m=0; ch1=0;
	
				while(!(m&0x8000))
				{
					pci_read_config_word(dev, 0x4e, (u16*)&m);
					ch1++;
					if(ch1==0xfff)
					{
						ch1=0;
						pci_write_config_word(dev, 0x4e, 0xd0);
					}
				};
	
				pci_read_config_dword(dev, 0x50, (u32 *)&sss);
 
     				SN[2]=sss&0xff;
     				SN[3]=(sss>>8)&0xff;
     				SN[4]=(sss>>16)&0xff;
				aDeviceInfo[number].bSN =(SN[0]-0x30)*10000+(SN[1]-0x30)*1000+(SN[2]-0x30)*100+(SN[3]-0x30)*10+(SN[4]-0x30);
 
				number++;

			}
		}
 	    	dev = pci_get_device(VID,PID, dev);
 	}
	if(!number)
	{
		printk ("%s Sorry, devices were not found\n  initialization failed\n", driverNameStr);
		return -1;
	}


	result = alloc_chrdev_region(&firstDevNum,0,number,driverNameStr);
	if(result<0)
	{
		printk ("%s alloc_chrdev_region failed\n", driverNameStr);
		return -1;
	}
		
	devClass = class_create(THIS_MODULE,classNameStr);
	
	if(devClass==NULL)
	{
		printk ("%s class_create failed\n", driverNameStr);
		unregister_chrdev_region(firstDevNum,1);
		return -1;
	}
	
	
	for(i=0; i<number; i++)
	{
	
		currentDevNum = MKDEV(MAJOR(firstDevNum),MINOR(firstDevNum)+i);
		aDeviceInfo[i].currentDevNum = currentDevNum;
			
		newDev = device_create(devClass,NULL,currentDevNum,NULL,"%s%d",deviceNameStr,i);

		if(newDev==NULL)
		{
			printk ("%s device_create failed\n", driverNameStr);
			class_destroy(devClass);	
			unregister_chrdev_region(currentDevNum,1);
			return -1;
		}

		cdev_init(&aDeviceInfo[i].cDev, &thisModule_fops);
		
		if(cdev_add(&aDeviceInfo[i].cDev,currentDevNum,1)==-1)
		{
			printk ("%s cdev_add failed\n", driverNameStr);
			device_destroy(devClass,currentDevNum);	
			class_destroy(devClass);	
			unregister_chrdev_region(currentDevNum,1);
			return -1;
		}
		if(!i)
			printk ("%s initialization\n", driverNameStr);
		printk ("           /dev/%s%d  serial# %d\n",deviceNameStr,i,aDeviceInfo[i].bSN);
			
	}

	for (i=0; i<number; i++)
		 aDeviceInfo[i].busy= 0;

	for (i=number; i<DEVICE_NUMBER; i++)
		 aDeviceInfo[i].busy = 2;
  
 	return 0; 
}

//================================================================================

static void __exit thisModule_cleanup(void) 
 {  
	 int i;
	 for(i=0; i<DEVICE_NUMBER; i++)
	 {
		if(aDeviceInfo[i].busy != 2)
		{
			cdev_del(&aDeviceInfo[i].cDev);
			device_destroy(devClass,aDeviceInfo[i].currentDevNum);	
		}
	 }
	class_destroy(devClass);	
	unregister_chrdev_region(firstDevNum,1);

	printk ("%s cleanup\n",driverNameStr);
   }
 
//================================================================================

module_init(thisModule_initialization);
module_exit(thisModule_cleanup);
